/************************************************************************
 * DESCR:	implements the DMK disk format, it gets read into the
 *		appropriate structure.
 *
 * NOTES:	
 ************************************************************************/
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include "standard.h"
#include "dmk.h"
#include "floppy.h"
#include "error.h"

/************************************************************************
 * NAME:	dmk_report()
 *
 * DESCR:	Generates a report of the (apparently) dmk file.
 *
 * ARGS:	level refers to the amount of reporting, 1 is lowest,
 *		3 is highest.
 *
 * RETURNS:	
 *
 * NOTES:	- levels other than 1 cause a detailed look at the file
 *		  beyond the guessing that occurred before
 ************************************************************************/
int
dmk_report(int fd, struct floppy *floppy, int level)
{
    int	errors;

    printf("DMK");

    if (level == 1) {		/* first level, just indicate the type	*/
	printf("\n");
    }

    if (level == 2 || level == 3 || level == 4) {
	lseek(fd,(off_t)0,SEEK_SET);
	errors = dmk_read(fd,floppy);
	floppy_report(floppy,level,errors);
    }
}

/************************************************************************
 * NAME:	dmk_guesser()
 *
 * DESCR:	Given a fd, tries to guess if it is a dmk file.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if it thinks the fd is a DMK file
 *
 * NOTES:	This routine is free to consume whatever it wants on the
 *		fd, so the caller may want to rewind afterwards.
 ************************************************************************/
int
dmk_guesser(int fd)
{
    int			errors;
    struct floppy	floppy;

    floppy_init(&floppy);
    errors = dmk_read(fd,&floppy);

    /* catch some common false positives	*/

    if (floppy.tracks < 1 || floppy.sectors < 1) {
	return(FALSE);
    }

    floppy_free(&floppy);

    if (errors != E_NONE) {
	return(FALSE);
    } else {
	return(TRUE);
    }
}

static int 
min(int a, int  b)
{
    if (a <= b)
	return(a);
    else
	return(b);
}

/************************************************************************
 * NAME:	dmk_read()
 *
 * DESCR:	Reads a dmk format disk into the DMK structure.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int dmk_read(int fd, struct floppy *floppy)
{
    unsigned char	c;
    int			i;
    int			side;
    int			track;
    int			range;
    unsigned char	*tracktmp;	/* storage of track data	*/
    int			track_length;	/* length of DMK data		*/
    int			options;	/* dmk header options field	*/
    int			sectors;	/* temporary sector count	*/
    int			sector_ptr;	/* index into track array for secs */
    int			dataptr;
    struct sector	*this_sector;	/* just to make coding easier	*/
    encoding		storage_density;

    /* read and process the header - header is 16 bytes	*/

    read(fd,&c,(size_t)1);	floppy->write_protect = (int)c;
    read(fd,&c,(size_t)1);	floppy->tracks = (int) c;
    read(fd,&c,(size_t)1);	track_length = (int)c;
    read(fd,&c,(size_t)1);	track_length += (int)c * 256;
    read(fd,&c,(size_t)1);	options = (int)c;

    if (floppy->write_protect != 0x00 && floppy->write_protect != 0xff) {
	return(E_FATAL|E_DMK_WP);
    }

    /* padding of 7 bytes	*/

    read(fd,&c,(size_t)1);
    read(fd,&c,(size_t)1);
    read(fd,&c,(size_t)1);
    read(fd,&c,(size_t)1);
    read(fd,&c,(size_t)1);
    read(fd,&c,(size_t)1);
    read(fd,&c,(size_t)1);

    /* must have zeros here or be bad					*/

    i = 0;
    read(fd,&c,(size_t)1);	i |= c;
    read(fd,&c,(size_t)1);	i |= c;
    read(fd,&c,(size_t)1);	i |= c;
    read(fd,&c,(size_t)1);	i |= c;

    if ( i != 0 ) {
	return(E_FATAL|E_DMK_ZEROS);
    }

    floppy->sectors = 0;			/* unknown for now	*/
    floppy->sides = (DMK_SIDE(options))?1:2;
    storage_density = (DMK_DENSITY(options))?WD_FM:WD_MFM;
    floppy->encoding = (DMK_RX02(options))?RX02:storage_density;

    /* prepare for the track data storage	*/

    tracktmp = (char *)malloc(track_length);	/* skpping error condition	*/

#define CONSTRAIN_DP(ptr)	(min(ptr,track_length-1))

    /* initialize the track and sector storage	*/

    for (side=0; side <= 1; side++) {

	floppy->side[side] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

	for (i=floppy->tracks; i--; ) {
	    floppy->side[side][i].sectors = 0;
	    floppy->side[side][i].sector = (struct sector *)NULL;
	}
    }

    /* now read in all of the sectors, via tracks and sides	*/

    for (track=0; track < floppy->tracks; track++) {

	for (side=0; side < floppy->sides; side++) {

#ifdef NOTDEF
	    M("reading, ");
#endif

	    if ( read(fd,tracktmp,(size_t)track_length) == track_length) {

		/* loop through the sectors in this track	*/
		/* noting the approx locations of the data	*/

		sectors = 0;
		sector_ptr = 0;

		while(1) {

		    dataptr = tracktmp[CONSTRAIN_DP(sector_ptr++)];
		    dataptr += tracktmp[CONSTRAIN_DP(sector_ptr++)] * 256;

		    if (dataptr == 0 || dataptr == 0xffff) {
			break;		/* move on to next side or track*/
		    }

		    /* dataptr points to the sector data 	*/

		    floppy->side[side][track].sectors = ++sectors;

		    floppy->side[side][track].sector = 
			(struct sector *)realloc(floppy->side[side][track].sector,
						 sizeof(struct sector)*sectors);

#define THIS_SECTOR	floppy->side[side][track].sector[sectors-1]

		    THIS_SECTOR.encoding = (DMK_DDEN(dataptr)?WD_MFM:WD_FM);

		    dataptr = DMK_FLAGMASK(dataptr);	/* trim dataptr	*/

		    /* dataptr points to the spot on the track where the
		       data starts...use it to index into the track		*/

		    /* Decode an id block if idamp is valid */

		    if (tracktmp[CONSTRAIN_DP(dataptr)] == 0xfe) {	/* got a sector	*/

/* do an extra increment if required by the DMK storage density	*/

#define INC	dataptr+=(storage_density==WD_MFM && THIS_SECTOR.encoding!=WD_MFM)?2:1

			INC; THIS_SECTOR.id = tracktmp[CONSTRAIN_DP(dataptr)];
			INC; THIS_SECTOR.side = tracktmp[CONSTRAIN_DP(dataptr)];
			INC; THIS_SECTOR.sector = tracktmp[CONSTRAIN_DP(dataptr)];
			INC; THIS_SECTOR.sizecode = tracktmp[CONSTRAIN_DP(dataptr)];
			                 THIS_SECTOR.size = 128 << (THIS_SECTOR.sizecode & 3);
			INC; THIS_SECTOR.headCRC = tracktmp[CONSTRAIN_DP(dataptr)];
			INC; THIS_SECTOR.headCRC += tracktmp[CONSTRAIN_DP(dataptr)] * 256;
			INC;
#ifdef NOTDEF
			M3("track %d, side %d, Lsector %d, ",track,side,sectors);
			M2("encoding %d (overall %d), ",THIS_SECTOR.encoding,storage_density);
			M4("id %d, side %d, sector %d, sizecode %d, ", 
			   THIS_SECTOR.id,
			   THIS_SECTOR.side,
			   THIS_SECTOR.sector,
			   THIS_SECTOR.sizecode);
#endif
			/* Project where DAM limit is */

			if (THIS_SECTOR.encoding == WD_FM) {
			    if (storage_density == WD_MFM) {
				range = 60;
			    } else {
				range = 30;   /* dataptr doesn't move */
			    }
			} else {
			    range = 43;
			}
#ifdef NOTDEF
			M("progress, ");
#endif
			while (range--) {
			    int dam = tracktmp[CONSTRAIN_DP(dataptr)];
			    INC;

			    if (dam >= 0xf8 && dam <= 0xfb) {
				THIS_SECTOR.mark = dam;
				break;
			    }
			}
#ifdef NOTDEF
			M1("progress (range %d), ", range);
#endif

			if (range < 0) {
			    return(E_FATAL|E_DMK_DAM);
			}
#ifdef NOTDEF
			M("progress, ");
#endif

			for (i=0; i < THIS_SECTOR.size; i++) {
			    THIS_SECTOR.data[min(i,sizeof(THIS_SECTOR.data))] = tracktmp[CONSTRAIN_DP(dataptr)];
			    INC;
			}
#ifdef NOTDEF
			M("copied\n");
#endif

			THIS_SECTOR.dataCRC = tracktmp[CONSTRAIN_DP(dataptr)];
			INC;
			THIS_SECTOR.dataCRC += tracktmp[CONSTRAIN_DP(dataptr)] * 256;
		    }
		}
/*
	    fprintf(stderr,"%d %d %d %d %d %d %d %d %d %d \n", 
		    THIS_SECTOR.id,
		    THIS_SECTOR.side,
		    THIS_SECTOR.sector,
		    THIS_SECTOR.sizecode,
		    THIS_SECTOR.size,
		    THIS_SECTOR.mark,
		    THIS_SECTOR.headCRC,
		    THIS_SECTOR.dataCRC,
		    THIS_SECTOR.length,
		    THIS_SECTOR.encoding);
*/
	    }
	}
    }

    free(tracktmp);

    /* at this point, set floppy->sectors to the maximum number	*/

    for (track=0; track < floppy->tracks; track++) {
	for (side=0; side < floppy->sides; side++) {
	    if (floppy->sectors < floppy->side[side][track].sectors) {
		floppy->sectors = floppy->side[side][track].sectors;
	    }
	}
    }

    /* now take a look at sector density and set the topical	*/
    /* density appropriately...					*/

    {
	int mixed;

	floppy->encoding = floppy_overall_encoding(floppy, &mixed, -1);
    }

    return(E_NONE);
}

/************************************************************************
 * NAME:	dmk_dump_bytes()
 *
 * DESCR:	A routine that dumps the given number of the given byte.
 *		Quite useful for gap generation.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- takes into account the density mode as well as the
 *		  density of the current sector.
 * if were are in exclusive SD mode, then we write one byte per byte.  If 
 * we are in mixed mode, but the sector is SD, the two bytes per byte are
 * written.	
 ************************************************************************/
dmk_dump_bytes(struct floppy	*floppy,
	       encoding		 encoding,
	       unsigned char	*buffer,
	       int		 count,
	       unsigned char	 c)
{
    int			 offset;
    unsigned char	*ptr = buffer;

    /* note that this code anticipates the BLANK_SECTOR encoding, in	*/
    /* BLANK_SECTORs should be written in "single" fashion		*/

    if (floppy->encoding != WD_FM && encoding == WD_FM ) {
	while (count--) {
	    *ptr++ = c;
	    *ptr++ = c;
	}
    } else {
	while (count--) {
	    *ptr++ = c;
	}
    }

    return(ptr-buffer);
}

/************************************************************************
 * NAME:	dmk_dump_sector()
 *
 * DESCR:	Dumps a sector in DMK format.
 *
 * ARGS:	
 *
 * RETURNS:	the amount of space consumed in the buffer.  If -1, is
 *		returned, then an error occurred (normally a sector with
 *		an unsupported encoding).  If 0 is returned, then something
 *		was done, but no IDAM is available (ie - the sector is
 *		skipped as a sector - this is used for blanks)  Also, the
 *		idam_offset pointer is filled in with a relative offset
 *		to the IDAM within the sector.  If this
 *		is zero, then the IDAM starts at the beginning of the buffer.
 *		If this is greater than the given buffer length, then an error
 *		occurred.
 *
 * NOTES:	- much of this code is based upon jv2cmk.c
 ************************************************************************/
int
dmk_dump_sector(struct floppy	*floppy,
		int		 track,
		int		 side,
		int		 sector,
		unsigned char	*trackbuf,
		int		 buflen,
		int		*idam_offset)
{
    int			 i;
    unsigned char	*ptr = trackbuf;

    *idam_offset = 0;			/* planning ahead */

#define DUMP(count,byte)  (dmk_dump_bytes(floppy,floppy->side[side][track].sector[sector].encoding,ptr,count,byte))

    /* note that unlike the jv2dmk.c code, this doesn't plan for IBM index mark */

    /* unlike what was done in the jv2dmk.c code, this code sets the gaps to
       very static values.  Probably not a great thing to do, but it is the same
       thing that is done in the firmware.					*/

    switch (floppy->side[side][track].sector[sector].encoding) {

        case WD_FM:
	    ptr += DUMP(14, 0xff);
	    ptr += DUMP(6, 0x00);
	    *idam_offset = (ptr-trackbuf);
	    break;

        case WD_MFM:
	    ptr += DUMP(22, 0x4e);
	    ptr += DUMP(12, 0x00);
	    ptr += DUMP(3, 0xa1);
	    *idam_offset = (ptr-trackbuf) | 0x8000;			/* flag for DD	*/
	    break;

	case BLANK_SECTOR:
	    /* just pad to an appropriate size depending upon the overall encoding */
	    switch (floppy->encoding) {
	        case WD_FM:
		    ptr += DUMP(256+58, 0xff);
		    break;
	        case WD_MFM:
		    ptr += DUMP(256+100, 0x4e);
		    break;
	        default:
		    return(-1);	/* encoding not supported, error return	*/
	    }
	    return(0);		/* no error, but no sector, idam_offset not valid	*/

	default:
	    return(-1);		/* encoding not supported, error return	*/
    }

    ptr += DUMP(1, 0xfe);
    ptr += DUMP(1, floppy->side[side][track].sector[sector].id);
    ptr += DUMP(1, side);
    ptr += DUMP(1, floppy->side[side][track].sector[sector].sector);
    ptr += DUMP(1, floppy->side[side][track].sector[sector].sizecode);
    ptr += DUMP(1, floppy->side[side][track].sector[sector].headCRC & 0xff);
    ptr += DUMP(1, floppy->side[side][track].sector[sector].headCRC / 256 & 0xff);

    switch (floppy->side[side][track].sector[sector].encoding) {
        case WD_FM:
	    ptr += DUMP(11, 0xff);
	    ptr += DUMP(6, 0x00);
	    break;

        case WD_MFM:
	    ptr += DUMP(20, 0x4e);
	    ptr += DUMP(12, 0x00);
	    ptr += DUMP(3, 0xa1);
	    break;
    }

    ptr += DUMP(1, floppy->side[side][track].sector[sector].mark);

    for (i=0; i < floppy->side[side][track].sector[sector].size; i++) {
	ptr += DUMP(1, floppy->side[side][track].sector[sector].data[i]);
    }
	
    ptr += DUMP(1, floppy->side[side][track].sector[sector].dataCRC & 0xff);
    ptr += DUMP(1, floppy->side[side][track].sector[sector].dataCRC / 256 & 0xff);

    switch (floppy->side[side][track].sector[sector].encoding) {
        case WD_FM:
	    ptr += DUMP(8, 0xff);
	    break;

        case WD_MFM:
	    ptr += DUMP(10, 0x4e);
	    break;
    }

    return(ptr-trackbuf);
}

/************************************************************************
 * NAME:	dmk_dump_track()
 *
 * DESCR:	Write a track to the buffer in dmk format.
 *
 * ARGS:	
 *
 * RETURNS:	the number of sectors processed.  If less than expected,
 *		then an error occurred on the last processed.
 *
 * NOTES:	
 ************************************************************************/
int
dmk_dump_track(struct floppy	*floppy,
	       int		 track,
	       int		 side,
	       unsigned char	*trackbuf,
	       int		 buflen)
{
#define HDR_OFFSET	0x80

    unsigned char	*sectbuf;
    int			 sectbuflen;
    int			 idam_offset;
    int			 i;
    int			 space_consumed;
    int			 idam_ptr = 0;

    sectbuf = trackbuf + HDR_OFFSET;
    sectbuflen = buflen - HDR_OFFSET;

    for (i=0; i < floppy->side[side][track].sectors; i++) {

	space_consumed = dmk_dump_sector(floppy, track, side, i, sectbuf, sectbuflen, &idam_offset);

	if (space_consumed == -1) {
	    break;		/* an error occurred	*/
	}
	if (space_consumed != 0) {
	    if (idam_offset > 0) { 
		idam_offset = DMK_FLAGMASK(idam_offset + (sectbuf - trackbuf));
		if (floppy->side[side][track].sector[i].encoding == WD_MFM) {
		    idam_offset = DMK_DDEN_SET(idam_offset);
		}
		trackbuf[idam_ptr*2] = idam_offset & 0xff;
		trackbuf[idam_ptr*2+1] = (idam_offset/256) & 0xff;
		idam_ptr++;
	    }
	}
	sectbuf += space_consumed;
	sectbuflen -= space_consumed;

	if (sectbuf > trackbuf + buflen) {
	    return(i);
	}
    }

    /* need to clear the rest of the IDAM pointers	*/

    for (; idam_ptr < 64; idam_ptr++) {
	trackbuf[idam_ptr*2] = 0;
	trackbuf[idam_ptr*2+1] = 0;
    }

    return(i);
}
	

/************************************************************************
 * NAME:	dmk_dump()
 *
 * DESCR:	Dump out a floppy in DMK format.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	- as noted in many areas, the concept for this code for
 *		  DMK dumping was from cw2dmk-2.7b/jv2dmk.c. by Tim Mann.
 ************************************************************************/
int
dmk_dump(int fd, struct floppy *floppy)
{
    unsigned char	c;
    int			options;
    int			track_length;
    int			sectors;
    int			i, j;
    unsigned char      *tracktmp;

    /* the following track lengths were chosen to match those used by the DMK	*/
    /* code in jv2dmk.c.  These COULD be calculated.  See that doc for more info*/
    /*										*/
    /*  0x1900 - double density 5.25"   (6400) - 18 sectors			*/
    /*	0x0cc0 - single density 5.25"   (3264) - 10 sectors			*/
    /*  0x14e0 - single density 8"      (5344) - 16? sectors (TODO)		*/
    /*  0x2940 - double density 8"      (10560) - 30? sectors (TODO)		*/

    switch (floppy->sectors) {
       case 10:		track_length = 0x0cc0; break;
       case 18:		track_length = 0x1900; break;
       case 16:		track_length = 0x14e0; break;
       case 30:		track_length = 0x2940; break;
       default:		track_length = floppy->sectors * 350;
    }

    /* the density is set by the "floppy->encoding" which should take care of
       mixed floppies as well as uniform density floppies			*/

    options = DMK_SIDE_SET(floppy->sides) | DMK_DENSITY_SET(floppy->encoding);

    /* write out the DMK header	*/

    c = (floppy->write_protect!=0)?0xff:0x00;	write(fd,&c,(size_t)1);
    c = floppy->tracks;				write(fd,&c,(size_t)1);
    c = track_length & 0xff;			write(fd,&c,(size_t)1);
    c = (track_length / 256) & 0xff;		write(fd,&c,(size_t)1);
    c = options;				write(fd,&c,(size_t)1);

    /* rest of header is padded with zeros	*/

    for (i=11, c=0; i--; ) {
	write(fd,&c,(size_t)1);
    }

    /* prepare for the track data storage, time to scribble out track by track	*/
    /* each track has a pointer to the IDAM of the track - which includes an	*/
    /* offset for the track header						*/

    tracktmp = (char *)malloc(track_length);	/* skpping error condition	*/

    for (i=0; i < floppy->tracks; i++) {
	for (j=0; j < floppy->sides; j++) {
	    sectors = dmk_dump_track(floppy,i,j,tracktmp,track_length);
	    write(fd,tracktmp,(size_t)track_length);
	    if (sectors != floppy->side[j][i].sectors) {
		return(FALSE);
	    }
	}
    }

    return(TRUE);
}


/************************************************************************
 * NAME:	dmk_init()
 *
 * DESCR:	Registers the dmk floppy format.
 *
 * ARGS:	
 *
 * RETURNS:	TRUE if the registration went OK, FALSE otherwise.
 *
 * NOTES:	- there is no WRITE for dmk at this time.
 ************************************************************************/
int
dmk_init()
{
    if (!floppy_format_register("DMK", "TRS-80 DMK",dmk_guesser, dmk_read, dmk_dump, dmk_report )) {
	return(FALSE);
    }

    if (!floppy_fileext_register("dmk", EXT_VERY_SPECIFIC, "DMK")) {
	return(FALSE);
    }

    if (!floppy_fileext_register("dsk", EXT_VERY_SPECIFIC, "DMK")) {
	return(FALSE);
    }

    return(TRUE);

}
